/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <EGL/egl.h>
#include <GLES2/gl2.h>
#include "WLEGLSurface.h"
#include "WLTouchRenderer.h"
#include <math.h>
#include <map>

#define PI (3.141592653589793)

#define WL_UNUSED(A) (A)=(A)

#define TOUCH_CIRCLE_POINTS 21
#define RADIUS 20

extern int gNeedRedraw;


// Touch event handler
static void  TouchHandleDown(void*, struct wl_touch*, uint32_t, uint32_t, struct wl_surface*,
                     int32_t, wl_fixed_t, wl_fixed_t);
static void  TouchHandleUp(void*, struct wl_touch*, uint32_t, uint32_t, int32_t);
static void  TouchHandleMotion(void*, struct wl_touch*, uint32_t, int32_t, wl_fixed_t, wl_fixed_t);
static void  TouchHandleFrame(void*, struct wl_touch*);
static void  TouchHandleCancel(void*, struct wl_touch*);

const struct wl_touch_listener TouchListener = {
    TouchHandleDown,
    TouchHandleUp,
    TouchHandleMotion,
    TouchHandleFrame,
    TouchHandleCancel,
};


//////////////////////////////////////////////////////////////////////////////
struct TouchPoint
{
    GLfloat x;
    GLfloat y;
};

typedef std::map<int32_t, TouchPoint>   TouchPointMap;

static TouchPointMap* g_pTouchPointMap = NULL;

TouchPointMap* GetTouchPointMap()
{
    if (!g_pTouchPointMap)
        g_pTouchPointMap = new TouchPointMap();

    return g_pTouchPointMap;
}


static void 
TouchHandleDown(void* data, struct wl_touch* touch, uint32_t serial, uint32_t time,
                struct wl_surface* surface, int32_t id, wl_fixed_t xw, wl_fixed_t yw)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    WL_UNUSED(surface);
    float x = (float)wl_fixed_to_double(xw);
    float y = (float)wl_fixed_to_double(yw);

    TouchPointMap* pTouchPointMap =  GetTouchPointMap();
    
    if (pTouchPointMap)
    {
        TouchPoint& rTP = (*pTouchPointMap)[id];
        rTP.x = x;
        rTP.y = y;
    }
    
    gNeedRedraw = 1;
    printf("ENTER EGLWLINPUT TouchHandleDown id: %d, x: %f,y: %f \n", id, x, y);
}

static void 
TouchHandleUp(void* data, struct wl_touch* touch, uint32_t serial, uint32_t time, int32_t id)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(serial);
    WL_UNUSED(time);
    
    TouchPointMap* pTouchPointMap =  GetTouchPointMap();
    
    if (pTouchPointMap)
        pTouchPointMap->erase(id);
    
    printf("ENTER EGLWLINPUT TouchHandleUp id: %i\n", id);
}

static void 
TouchHandleMotion(void* data, struct wl_touch* touch, uint32_t time, int32_t id,
                  wl_fixed_t xw, wl_fixed_t yw)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
    WL_UNUSED(time);
    float x = (float)wl_fixed_to_double(xw);
    float y = (float)wl_fixed_to_double(yw);

    TouchPointMap* pTouchPointMap =  GetTouchPointMap();
    
    if (pTouchPointMap)
    {
        TouchPoint& rTP = (*pTouchPointMap)[id];
        rTP.x = x;
        rTP.y = y;
    }
    printf("ENTER EGLWLINPUT TouchHandleMotion id: %d, x: %f,y: %f \n", id, x, y);
    gNeedRedraw = 1;
}

static void 
TouchHandleFrame(void* data, struct wl_touch* touch)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
}

static void 
TouchHandleCancel(void* data, struct wl_touch* touch)
{
    WL_UNUSED(data);
    WL_UNUSED(touch);
}

//////////////////////////////////////////////////////////////////////////////

void  WaitForEvent(struct wl_display* wlDisplay)
{
    wl_display_dispatch(wlDisplay);
}

//////////////////////////////////////////////////////////////////////////////

typedef struct _es2shaderobject {
    GLuint fragmentShaderId;
    GLuint vertexShaderId;
    GLuint shaderProgramId;
    GLint  matrixLocation;
    GLint  colorLocation;
} ES2ShaderObject;

typedef struct _es2vertexbufferobject {
    GLuint vbo;
} ES2VertexBuffer;

static ES2ShaderObject	gShaderObject;
static ES2VertexBuffer	gVertexBuffer;

// Fragment shader code
const char* sourceFragShader = "  \
    uniform mediump vec4 u_color; \
    void main(void)               \
    {                             \
        gl_FragColor = u_color;   \
    }";

// Vertex shader code
const char* sourceVertexShader = " \
    attribute highp vec4 a_vertex; \
    uniform mediump mat4 u_matrix; \
    void main(void)                \
    {                              \
        gl_Position = u_matrix * a_vertex; \
    }";

//////////////////////////////////////////////////////////////////////////////

static bool InitShader();
static bool InitVertexBuffer();


float world_mtx[16] = { 1.0,  0.0,  0.0,  0.0,
                        0.0,  1.0,  0.0,  0.0,
                        0.0,  0.0,  1.0,  0.0,
                       -1.0, -1.0,  0.0,  1.0};

int x_offset = 0;
int y_offset = 0;


bool
InitRenderer(int width, int height)
{
    glViewport(0, 0, width, height);

    if (!InitShader()){
        return false;
    }

    if (!InitVertexBuffer()){
        return false;
    }

    world_mtx[0] = 2.0/width;
    world_mtx[5] = 2.0/height;
    x_offset = width/2;
    y_offset = height/2;

    glClearColor(0.2, 0.2, 0.2, 1.0);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
    return true;
}


void
TerminateRenderer()
{
    if (g_pTouchPointMap)
    {
        delete g_pTouchPointMap;
        g_pTouchPointMap = NULL;
    }

    glDeleteProgram(gShaderObject.shaderProgramId);
    glDeleteShader(gShaderObject.fragmentShaderId);
    glDeleteShader(gShaderObject.vertexShaderId);

    glDeleteBuffers(1, &gVertexBuffer.vbo);
}

char*
BuildShaderErrorLog(GLuint id)
{
    int l, nChar;
    glGetShaderiv(id, GL_INFO_LOG_LENGTH, &l);

    char* info = (char*)malloc(sizeof(char) * l);
    glGetShaderInfoLog(id, l, &nChar, info);

    return info;
}

static bool
InitShader()
{
    GLint result = 0;
    char* log = NULL;

    // Create the fragment shader object
    gShaderObject.fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
    glShaderSource(gShaderObject.fragmentShaderId, 1,
        (const char**)&sourceFragShader, NULL);
    glCompileShader(gShaderObject.fragmentShaderId);

    glGetShaderiv(gShaderObject.fragmentShaderId, GL_COMPILE_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.fragmentShaderId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    // Create the vertex shader object
    gShaderObject.vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
    glShaderSource(gShaderObject.vertexShaderId, 1,
        (const char**)&sourceVertexShader, NULL);
    glCompileShader(gShaderObject.vertexShaderId);

    glGetShaderiv(gShaderObject.vertexShaderId, GL_COMPILE_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.vertexShaderId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    gShaderObject.shaderProgramId = glCreateProgram();

    glAttachShader(gShaderObject.shaderProgramId, gShaderObject.fragmentShaderId);
    glAttachShader(gShaderObject.shaderProgramId, gShaderObject.vertexShaderId);

    glBindAttribLocation(gShaderObject.shaderProgramId, 0, "a_vertex");

    glLinkProgram(gShaderObject.shaderProgramId);

    glGetProgramiv(gShaderObject.shaderProgramId, GL_LINK_STATUS, &result);
    if (!result){
        log = BuildShaderErrorLog(gShaderObject.shaderProgramId);
        fprintf(stderr, "Failed to compile fragment shader: %s\n", log);
        if (log) free(log);
        return false;
    }

    glUseProgram(gShaderObject.shaderProgramId);
    gShaderObject.matrixLocation = glGetUniformLocation(
        gShaderObject.shaderProgramId, "u_matrix");
    gShaderObject.colorLocation = glGetUniformLocation(
        gShaderObject.shaderProgramId, "u_color");

    return true;
}

static bool
InitVertexBuffer()
{
    glGenBuffers(1, &gVertexBuffer.vbo);
    glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);

    GLfloat data[TOUCH_CIRCLE_POINTS*2];
    
    data[0] = data[1] = 0.0f;
    
    for(unsigned int n = 0; n < TOUCH_CIRCLE_POINTS*2; n += 2)
    {
        GLfloat angle = (GLfloat)(PI / (TOUCH_CIRCLE_POINTS-2)) * n;
        data[n+2] = RADIUS * sin(angle);
        data[n+3] = RADIUS * cos(angle);
    }

    for(unsigned int n = 0; n < TOUCH_CIRCLE_POINTS*2; n += 2)
        printf("x:%f y:%f\n", data[n], data[n+1]);

    unsigned int uiSize = TOUCH_CIRCLE_POINTS * (sizeof(GLfloat) * 2);
    glBufferData(GL_ARRAY_BUFFER, uiSize, data, GL_STATIC_DRAW);

    return true;
}

static void 
AttachVertexBuffer()
{
    glBindBuffer(GL_ARRAY_BUFFER, gVertexBuffer.vbo);
    glEnableVertexAttribArray(0);
    glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, 0, 0);
}

static void 
DetachVertexBuffer()
{
    glBindBuffer(GL_ARRAY_BUFFER, 0);
}

static void
DrawCircle(float x, float y)
{
    static float white[] = {1.0, 1.0, 1.0, 1.0};

    world_mtx[12] = (x - x_offset) * world_mtx[0];
    world_mtx[13] = -(y - y_offset) * world_mtx[5];

    glUniformMatrix4fv(gShaderObject.matrixLocation, 1, GL_FALSE, world_mtx);
    glUniform4fv(gShaderObject.colorLocation, 1, white);

    glDrawArrays(GL_TRIANGLE_FAN, 0, TOUCH_CIRCLE_POINTS);
    glFinish();
}


bool
DrawTouches(WLEGLSurface* surface)
{
    glClear(GL_COLOR_BUFFER_BIT);
    glUseProgram(gShaderObject.shaderProgramId);

    AttachVertexBuffer();

    TouchPointMap* pTouchPointMap =  GetTouchPointMap();

    if (pTouchPointMap)
    {
        for(TouchPointMap::const_iterator it = pTouchPointMap->begin();
            it != pTouchPointMap->end(); it++)
        {
            const TouchPoint& rTP = it->second;
            DrawCircle(rTP.x, rTP.y);
        }

        fprintf(stderr, "Number of touches: %d\n", pTouchPointMap->size());
    }
    
    DetachVertexBuffer();
    eglSwapBuffers(surface->GetEGLDisplay(), surface->GetEGLSurface());

    return true;
}
